//
//  GSPipControllerManager.m
//  PIPDemo
//
//  Created by 230222 on 2025/11/14.
//

#import "GSPipControllerManager.h"
#import <AVKit/AVKit.h>
#import <AVFoundation/AVFoundation.h>

@interface GSPipControllerManager ()<AVPictureInPictureControllerDelegate>
@property (strong, nonatomic) AVPictureInPictureController *pipController;
@property (strong, nonatomic) AVPlayer *player;
@property (strong, nonatomic) AVPlayerLayer *playerLayer;
@property (nonatomic, strong) NSLock *lock;
@property (nonatomic, strong) CIImage *placeholderCIImage;
@property (nonatomic, strong) AVAsynchronousCIImageFilteringRequest *  request;
@property (nonatomic, strong) CIContext *context;
@property (nonatomic, strong) CIImage * blackBackgroundImage ;
@property (nonatomic, assign) CGSize targetSize;
@end


@implementation GSPipControllerManager
- (instancetype)initWith:(UIView *)playerView{
    if(self = [super init]){
        self.targetSize = CGSizeMake(1280, 720);
        self.lock = [[NSLock alloc] init];
        self.blackBackgroundImage = [self createBlackBackgroundImageTargetSize:self.targetSize];
        self.placeholderCIImage = self.blackBackgroundImage;

        self.context = [CIContext context];

        NSString *videoPath = [[NSBundle mainBundle] pathForResource:@"black_video" ofType:@"mp4"];
        NSURL *videoURL = [NSURL fileURLWithPath:videoPath];
        AVAsset *asset = [AVAsset assetWithURL:videoURL];
        
        // 创建视频合成
        AVMutableVideoComposition *videoComposition = [self createVideoCompositionWithAsset:asset];
        
        AVPlayerItem *playerItem = [AVPlayerItem playerItemWithAsset:asset];
        playerItem.videoComposition = videoComposition;
        
        self.player = [AVPlayer playerWithPlayerItem:playerItem];
        self.playerLayer = [AVPlayerLayer playerLayerWithPlayer:self.player];
        self.playerLayer.frame = playerView.bounds;
        [playerView.layer addSublayer:self.playerLayer];
        self.playerLayer.hidden = YES;
        // 创建 PIP 控制器
        self.pipController = [[AVPictureInPictureController alloc]
                             initWithPlayerLayer:self.playerLayer];

        self.pipController.delegate = self;
        // 隐藏快进快退
        if (@available(iOS 14.0, *)) {
            self.pipController.requiresLinearPlayback = YES;
        } else {
            // Fallback on earlier versions
        }
        [self setupPlaybackNotifications];

    }
    return self;
}
- (void)setPlayerLayerFrame:(CGRect)frame{
    self.playerLayer.frame = frame;
}
#pragma mark -- 开启画中画
- (void)startPip{
    NSLog(@"开启画中画");
    // 开始播放
    [self.player play];
}
#pragma mark -- 关闭画中画
- (void)stopPip{
    NSLog(@"关闭画中画");
    self.placeholderCIImage = self.blackBackgroundImage;
    // 暂停播放
    [self.player pause];
    if (self.pipController.pictureInPicturePossible) {
        if (self.pipController.isPictureInPictureActive) {
            [self.pipController stopPictureInPicture];
        }
    }else{
        NSLog(@"不能支持画中画");
    }
}
#pragma mark -- 设置占位图
- (void)resetPlaceholderCIImage:(CIImage *)placeholderCIImage{
    [self.lock lock];
    if(placeholderCIImage == nil){
        self.placeholderCIImage = self.blackBackgroundImage;
        NSLog(@"设置占位图 placeholderCIImage == nil");

    }else{
        self.placeholderCIImage = [self scaleCIImage:placeholderCIImage targetSize:self.targetSize blackBackground:self.blackBackgroundImage];
    }
    [self.lock unlock];
}
- (AVMutableVideoComposition *)createVideoCompositionWithAsset:(AVAsset *)asset {
    __weak typeof(self) weakSelf = self;
    // 创建视频合成
     AVMutableVideoComposition *videoComposition = [AVMutableVideoComposition videoCompositionWithAsset:asset applyingCIFiltersWithHandler:^(AVAsynchronousCIImageFilteringRequest * _Nonnull request) {
         dispatch_async(dispatch_get_main_queue(), ^{
             __strong typeof(weakSelf) strongSelf = weakSelf;
             @autoreleasepool {
                 strongSelf.request = request;
                 CIImage *sourceImage = strongSelf.request.sourceImage;
                 [strongSelf.lock lock];
                 if(strongSelf.placeholderCIImage == nil || sourceImage == nil ){
                     [strongSelf.request  finishWithError:[NSError errorWithDomain:AVFoundationErrorDomain
                                                                  code:AVErrorNoDataCaptured
                                                              userInfo:nil]];
                 }else{
                     [strongSelf.request finishWithImage:strongSelf.placeholderCIImage context:strongSelf.context];
                 }
                 [strongSelf.lock unlock];
             }
         });
     }];
     
     // 设置视频合成到playerItem
     return videoComposition;
}
#pragma mark --- 缩放到指定尺寸后，居中放到背景上。
- (CIImage *)scaleCIImage:(CIImage *)inputImage targetSize:(CGSize)size blackBackground:(CIImage *)blackBackground{
 
    CGRect extent = inputImage.extent;
    CGFloat scale = MIN(size.width / extent.size.width, size.height / extent.size.height);
    // 计算平移量使其居中
    CGFloat translateX = (size.width - extent.size.width * scale) / 2.0;
    CGFloat translateY = (size.height - extent.size.height * scale) / 2.0;
    
    // 先缩放后平移
    CGAffineTransform scaleTransform = CGAffineTransformMakeScale(scale, scale);
    CGAffineTransform translateTransform = CGAffineTransformMakeTranslation(translateX, translateY);
    CGAffineTransform finalTransform = CGAffineTransformConcat(scaleTransform, translateTransform);
    
    CIImage * scaledImage = [inputImage imageByApplyingTransform:finalTransform];
    
    // 4. 将缩放后的图像居中放置在背景上
    CIFilter *sourceOverCompositing = [CIFilter filterWithName:@"CISourceOverCompositing"];
    [sourceOverCompositing setValue:scaledImage forKey:kCIInputImageKey];
    [sourceOverCompositing setValue:blackBackground forKey:kCIInputBackgroundImageKey];
    
    return [sourceOverCompositing valueForKey:kCIOutputImageKey];
}
#pragma mark -- 创建黑色背景图像
- (CIImage *)createBlackBackgroundImageTargetSize:(CGSize)size{
    CIColor *blackColor = [CIColor colorWithRed:0 green:0 blue:0 alpha:1];
    CIFilter *constantColorGen = [CIFilter filterWithName:@"CIConstantColorGenerator"];
    [constantColorGen setValue:blackColor forKey:@"inputColor"];
    CIImage *blackBackground = [constantColorGen valueForKey:kCIOutputImageKey];
    blackBackground = [blackBackground imageByCroppingToRect:CGRectMake(0, 0, size.width, size.height)];
    return blackBackground;
}
#pragma mark -- 设置播放通知
- (void)setupPlaybackNotifications {
    // 添加播放完成通知
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(playerItemDidPlayToEndTime:)
                                                 name:AVPlayerItemDidPlayToEndTimeNotification
                                               object:nil];
    // 添加 rate 属性监听
    [self.player addObserver:self
                     forKeyPath:@"rate"
                        options:NSKeyValueObservingOptionNew
                        context:nil];
    
    
}
#pragma mark -- 播放结束通知回调
- (void)playerItemDidPlayToEndTime:(NSNotification *)notification {
    NSLog(@"播放结束通知回调");
    [self.player seekToTime:kCMTimeZero completionHandler:^(BOOL finished) {
        if (finished) {
            [self.player play];
        }
    }];
}
#pragma mark - KVO 监听
- (void)observeValueForKeyPath:(NSString *)keyPath
                      ofObject:(id)object
                        change:(NSDictionary<NSKeyValueChangeKey,id> *)change
                       context:(void *)context {
    
    if ([keyPath isEqualToString:@"rate"]) {
        float rate = [change[NSKeyValueChangeNewKey] floatValue];
        NSLog(@"播放速率：%f", rate);
        if (rate == 0.0) {
            // 播放暂停
            NSLog(@"播放暂停");
        } else if (rate == 1.0) {
            // 正常播放
            NSLog(@"开始播放");
            dispatch_async(dispatch_get_main_queue(), ^{
                if (self.pipController.pictureInPicturePossible) {
                    if (!self.pipController.isPictureInPictureActive) {
                        [self.pipController startPictureInPicture];
                    }
                }
            });
        } else if (rate == -1.0) {
            // 倒带播放
            NSLog(@"倒带播放");
        }
    }
}
- (void)dealloc {
    // 移除观察者
    [self.player removeObserver:self forKeyPath:@"rate"];
}
#pragma mark - AVPictureInPictureControllerDelegate

// PIP 即将启动
- (void)pictureInPictureControllerWillStartPictureInPicture:(AVPictureInPictureController *)pictureInPictureController {
    NSLog(@"画中画即将启动");
  
}

// PIP 已经启动
- (void)pictureInPictureControllerDidStartPictureInPicture:(AVPictureInPictureController *)pictureInPictureController {
    NSLog(@"画中画已经启动");
    self.playerLayer.hidden = NO;

}

// PIP 启动失败
- (void)pictureInPictureController:(AVPictureInPictureController *)pictureInPictureController
failedToStartPictureInPictureWithError:(NSError *)error {
    NSLog(@"画中画启动失败: %@", error.localizedDescription);
}

// PIP 即将停止
- (void)pictureInPictureControllerWillStopPictureInPicture:(AVPictureInPictureController *)pictureInPictureController {
    NSLog(@"画中画即将停止");

}

// PIP 已经停止
- (void)pictureInPictureControllerDidStopPictureInPicture:(AVPictureInPictureController *)pictureInPictureController {
    NSLog(@"画中画已经停止");
    self.playerLayer.hidden = YES;
    // 暂停播放
    [self.player pause];
    if(self.delegate && [self.delegate respondsToSelector:@selector(pipControllerDidStop)]){
        [self.delegate pipControllerDidStop];
    }
}

// PIP 恢复播放界面
- (void)pictureInPictureController:(AVPictureInPictureController *)pictureInPictureController
restoreUserInterfaceForPictureInPictureStopWithCompletionHandler:(void (^)(BOOL))completionHandler {
    NSLog(@"画中画恢复播放界面");
    completionHandler(YES);
}
@end
